<?php
/* --------------------------------------------------------------
  ConfigRepository.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit\Repositories;

use \StyleEdit\Entities\Interfaces\ConfigInterface;
use \StyleEdit\Entities\StoredConfig;

/**
 * Class ConfigRepository
 *
 * @package StyleEdit\Repositories
 */
class ConfigRepository
{
	/** @var \StyleEdit\Repositories\ConfigReader $reader */
	private $reader;
	
	/** @var \StyleEdit\Repositories\ConfigWriter $reader */
	private $writer;
	
	/** @var \StyleEdit\Repositories\ConfigDeleter $reader */
	private $deleter;
	
	
	/**
	 * ConfigRepository constructor.
	 *
	 * @param \StyleEdit\Repositories\ConfigReader  $reader
	 * @param \StyleEdit\Repositories\ConfigWriter  $writer
	 * @param \StyleEdit\Repositories\ConfigDeleter $deleter
	 */
	public function __construct(ConfigReader $reader, ConfigWriter $writer, ConfigDeleter $deleter)
	{
		$this->reader  = $reader;
		$this->writer  = $writer;
		$this->deleter = $deleter;
	}
	
	
	/**
	 * Saves a style configuration and returns the stored style configuration. A Config object will be
	 * inserted, a StoredConfig will be updated. The storing process ensures a unique name of the style
	 * configuration.
	 *
	 * @param \StyleEdit\Entities\Interfaces\ConfigInterface $styleConfig
	 * @param bool                                           $p_toOriginalTheme Write files to original theme directory.
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function store(ConfigInterface $styleConfig, $p_toOriginalTheme = false)
	{
		$this->_setUniqueStyleName($styleConfig);
		
		if($styleConfig instanceof StoredConfig)
		{
			return $this->writer->update($styleConfig, $p_toOriginalTheme);
		}
		
		return $this->writer->insert($styleConfig, $p_toOriginalTheme);
	}
	
	
	/**
	 * Returns a collection of all boilerplates.
	 *
	 * @return \StyleEdit\Collections\StoredBoilerplateConfigCollection
	 */
	public function getBoilerplates()
	{
		return $this->reader->getBoilerplates();
	}
	
	
	/**
	 * Returns a boilerplate identified by its name.
	 *
	 * @param string $p_boilerplateName
	 *
	 * @return \StyleEdit\Entities\StoredBoilerplateConfig
	 */
	public function getBoilerplateByName($p_boilerplateName)
	{
		return $this->reader->getBoilerplateByName($p_boilerplateName);
	}
	
	
	/**
	 * Returns a collection of all style configurations.
	 *
	 * @return \StyleEdit\Collections\StoredConfigCollection
	 */
	public function getStyleConfigs()
	{
		return $this->reader->getStyles();
	}
	
	
	/**
	 * Returns a style configuration identified by its name.
	 *
	 * @param string $p_styleName
	 *
	 * @return \StyleEdit\Entities\StoredConfig
	 */
	public function getStyleConfigByName($p_styleName)
	{
		return $this->reader->getStyleByName($p_styleName);
	}
	
	
	/**
	 * Deletes a style configuration identified by its name.
	 *
	 * @param string $p_styleName
	 * @param bool   $p_fromOriginalTheme Delete files from original theme directory.
	 *
	 * @return bool
	 */
	public function deleteStyleConfigByName($p_styleName, $p_fromOriginalTheme = false)
	{
		$storedStyleEditConfig = $this->reader->getStyleByName($p_styleName);
		
		return $this->deleter->delete($storedStyleEditConfig, $p_fromOriginalTheme);
	}
	
	
	/**
	 * Searches for an active style configuration and returns it if found. Otherwise null will be returned.
	 *
	 * @return \StyleEdit\Entities\StoredConfig|null
	 */
	public function findActiveStyleConfig()
	{
		return $this->reader->findActiveStyleConfig();
	}
	
	
	/**
	 * Creates an unique configuration name.
	 *
	 * The function will check if the given name is already used by an existing style configuration and will append it
	 * with an automated incremented integer wrapped in parentheses which is checked recursively until it is unique.
	 *
	 * @param ConfigInterface $styleConfig
	 *
	 * @return ConfigInterface $styleConfig
	 */
	private function _setUniqueStyleName(ConfigInterface $styleConfig)
	{
		$existingStylesArray = $this->getStyleConfigs()->toArray();
		$styleName           = $styleConfig->getName();
		
		if(!array_key_exists($styleName, $existingStylesArray)
		   || ($styleConfig instanceof StoredConfig
		       && $existingStylesArray[$styleName]->getFilename() === $styleConfig->getFilename())
		)
		{
			return $this;
		}
		
		preg_match('/(.+)\(([\d]+)?\)$/', $styleName, $matches);
		
		if(isset($matches[1]))
		{
			$newStyleName = $matches[1] . '(' . ($matches[2] + 1) . ')';
		}
		else
		{
			$newStyleName = $styleName . ' (1)';
		}
		
		$styleConfig->setName($newStyleName);
		
		if(array_key_exists($newStyleName, $existingStylesArray))
		{
			$this->_setUniqueStyleName($styleConfig);
		}
		
		return $this;
	}
}